
package service;

import java.io.FileInputStream;

import parser.parserTS;


/**
 * Implements the patter "Abstract Factory" http://en.wikipedia.org/wiki/Abstract_factory_pattern
 * that allows to create class that use the classes of the package service without 
 * that this classes have any references to the implementation of any class of this 
 * package.   
 */
public class ServiceFactoryImpl implements ServiceFactory {
	/**
	 * Return a new object of the class SimpleActionImpl
	 * @param name the name of the action
	 * @param from the source state of the action
	 * @param to the target state of the action
	 */
	public SimpleTransition createSimpleAction(String name, State from, State to) {
		return new SimpleTransitionImpl(name,from,to);
	}
	/**
	 * Return a new object of the class SimpleStateImpl
	 * @param name the name of the state
	 * @param isInitial this variable must be true if the state will be a initial state
	 * @param isFinal this variable must be true if the state will be a final state
	 */	
	public SimpleState createSimpleState(String name, boolean isInitial,
			boolean isFinal) {
		return new SimpleStateImpl(name,isInitial,isFinal);
	}
	/**
	 * Return a new object of the class ComposedActionImpl
	 * @param name the name of the action
	 * @param ts the transition system related to the ComposedAction
	 * @param from the source state of the action
	 * @param to the target state of the action
	 */
	public ComposedTransition createComposedAction(String name,
			TransitionSystem<? extends State, ? extends Transition> ts, ComposedState from, ComposedState to) {
		return new ComposedTransitionImpl(name,ts,from,to);
	}
	/**
	 * Return a new object of the class ComposedStateImpl
	 * @param name the name of the state
	 * @param states array of states that make up the ComposedState
	 */
	public ComposedState createComposedState(String name, State[] states) {
		return new ComposedStateImpl(name,states);
	}
	
	public ComposedState createComposedState(State[] states) {
		return new ComposedStateImpl(states);
	}
	/**
	 * Return a new object of the class TransitionSystemImpl that use ComposedState and ComposedAction
	 */
	public TransitionSystem<ComposedState, ComposedTransition> createComposedTransitionSystem() {
		return new TransitionSystemImpl<ComposedState, ComposedTransition>();
	}
	/**
	 * Return a new object of the class TransitionSystemImpl that use SimpleState and SimpleAction
	 */
	public TransitionSystem<SimpleState, SimpleTransition> createSimpleTransitionSystem() {
		return new TransitionSystemImpl<SimpleState, SimpleTransition>();
	}
	/**
	 * Return a new object of the class TransitionSystemImpl that use SimpleState and SimpleAction,
	 * it is initialized with the TransitionSystem contained in nameFile   
	 * @param nameFile the path of the file that contain the description of the TransitionSystem
	 */	
	public TransitionSystem<SimpleState, SimpleTransition> createSimpleTransitionSystem(String nameFile) throws Exception{
		TransitionSystem <SimpleState,SimpleTransition> TS=null;
		FileInputStream file=new FileInputStream(nameFile);
		ServiceFactory factory=new ServiceFactoryImpl();
		parserTS pars=new parserTS(file,factory);
		TS= pars.createTS();
		
		if(! TS.isValid())
			throw new Exception("TransitionSystem not valid: "+TS.getError());
		file.close();
		return TS;
	}
	

}
